<?php
require_once 'config.php';
require_once 'lib/ACMECert.php';

use skoerfgen\ACMECert\ACMECert;

session_start();

// --- SETTINGS HELPERS ---
function load_settings()
{
    $defaults = [
        'username' => APP_USERNAME,
        'password_hash' => APP_PASSWORD_HASH,
        'recovery_email' => '',
        'tracking_consent' => false,
        'recovery_code' => '',
        'recovery_expiry' => 0,
        'needs_password_change' => false
    ];
    if (!file_exists(SETTINGS_FILE)) {
        return $defaults;
    }
    $data = json_decode(file_get_contents(SETTINGS_FILE), true);
    return array_merge($defaults, $data ?: []);
}

function save_settings($settings)
{
    if (!is_dir(DATA_DIR))
        mkdir(DATA_DIR, 0755, true);
    file_put_contents(SETTINGS_FILE, json_encode($settings, JSON_PRETTY_PRINT));
}

$app_settings = load_settings();

// --- AUTH LOGIC ---
if (isset($_POST['login'])) {
    $user = $_POST['username'] ?? '';
    $pass = $_POST['password'] ?? '';

    // Check normal login or recovery code login
    if (
        ($user === $app_settings['username'] && password_verify($pass, $app_settings['password_hash'])) ||
        ($user === $app_settings['username'] && !empty($app_settings['recovery_code']) && $pass === $app_settings['recovery_code'] && time() < $app_settings['recovery_expiry'])
    ) {

        $_SESSION['authenticated'] = true;

        // If they used a recovery code, they MUST change password
        if ($pass === $app_settings['recovery_code']) {
            $app_settings['needs_password_change'] = true;
            $app_settings['recovery_code'] = ''; // Clear code
            save_settings($app_settings);
        }
    } else {
        $error = "Credenciales incorrectas o código expirado.";
    }
}

if (isset($_GET['logout'])) {
    session_destroy();
    header("Location: index.php");
    exit;
}

if (!isset($_SESSION['authenticated']) && !ENABLE_PUBLIC_ACCESS) {
    // --- RECOVERY LOGIC (6-DIGIT CODE) ---
    if (isset($_POST['recover_password'])) {
        $recovery_email = $app_settings['recovery_email'];
        if ($recovery_email) {
            $code = str_pad(rand(0, 999999), 6, '0', STR_PAD_LEFT);
            $app_settings['recovery_code'] = $code;
            $app_settings['recovery_expiry'] = time() + (15 * 60); // 15 minutes
            save_settings($app_settings);

            $to = $recovery_email;
            $subject = "Código de Recuperación - SSL FREE";
            $message = "Has solicitado recuperar tu acceso.\n\n";
            $message .= "Tu código de acceso temporal es: " . $code . "\n";
            $message .= "Este código expirará en 15 minutos.\n\n";
            $message .= "Usa este código como contraseña en la pantalla de inicio de sesión.\n";

            $headers = "From: noreply@" . $_SERVER['HTTP_HOST'] . "\r\n";
            if (@mail($to, $subject, $message, $headers)) {
                $status_message = "Se ha enviado un código de recuperación a: " . $recovery_email;
            } else {
                $status_error = "Error al enviar el correo. Contacta al soporte.";
            }
        } else {
            $status_error = "No hay un correo de recuperación configurado.";
        }
    }

    render_login($error ?? null, $status_message ?? '', $status_error ?? '');
    exit;
}

// --- MANDATORY SETUP CHECK ---
if ($app_settings['username'] === 'admin' || $app_settings['needs_password_change']) {
    if (isset($_POST['mandatory_setup'])) {
        $new_email = $_POST['setup_email'];
        $new_pass = $_POST['setup_password'];

        if (filter_var($new_email, FILTER_VALIDATE_EMAIL) && !empty($new_pass)) {
            $app_settings['username'] = $new_email;
            $app_settings['recovery_email'] = $new_email;
            $app_settings['password_hash'] = password_hash($new_pass, PASSWORD_BCRYPT);
            $app_settings['needs_password_change'] = false;
            save_settings($app_settings);
            $status_message = "Configuración completada. ¡Bienvenido!";
        } else {
            $setup_error = "Por favor ingresa un correo válido y una contraseña.";
        }
    }

    if ($app_settings['username'] === 'admin' || $app_settings['needs_password_change']) {
        render_mandatory_setup($setup_error ?? null);
        exit;
    }
}

// --- APP LOGIC ---
$status_message = "";
$certs = ['crt' => '', 'key' => '', 'bundle' => ''];

if (isset($_POST['generate_ssl'])) {
    try {
        $domain_input = $_POST['domain'] ?: (FORCE_DOMAIN ?: $_SERVER['HTTP_HOST']);
        $domains = array_map('trim', explode(',', $domain_input));
        $domain = $domains[0]; // Primary domain for display/logs
        $email = LE_EMAIL;
        $staging = !LE_PRODUCTION;

        $acme = new ACMECert(!$staging);

        // Load or Create Account Key
        $account_key_file = DATA_DIR . '/account.key';
        if (!file_exists($account_key_file)) {
            $account_key = $acme->generateRSAKey(4096);
            file_put_contents($account_key_file, $account_key);
            chmod($account_key_file, 0600);
        } else {
            $account_key = file_get_contents($account_key_file);
        }

        $acme->loadAccountKey($account_key);
        $acme->register(true, [$email]);

        // Generate Domain Key
        $domain_key = $acme->generateRSAKey(2048);

        // Challenge Handler
        $challenge_cb = function ($opts) {
            $path = $_SERVER['DOCUMENT_ROOT'] . $opts['key'];
            $dir = dirname($path);
            if (!is_dir($dir))
                mkdir($dir, 0755, true);
            file_put_contents($path, $opts['value']);
            return function () use ($path) {
                @unlink($path);
            };
        };

        // Get Certificate
        $chain = $acme->getCertificateChain($domain_key, $domains, $challenge_cb);

        // Split Chain (Leaf vs Bundle)
        $delim = '-----END CERTIFICATE-----';
        $parts = explode($delim, trim($chain));
        $certs['crt'] = trim($parts[0] . $delim);
        $certs['key'] = $domain_key;
        $certs['bundle'] = trim(implode($delim, array_slice($parts, 1)));

        // Try cPanel Auto-Install
        if (CPANEL_API_TOKEN && CPANEL_USER) {
            install_to_cpanel($domain, $certs['crt'], $certs['key'], $certs['bundle']);
            $status_message = "SSL Generado e Instalado exitosamente en cPanel!";
        } else {
            $status_message = "SSL Generado exitosamente. Por favor instalar manualmente.";
        }

        // --- NOTIFICATION ---
        $to = NOTIFICATION_EMAIL;
        $subject = "Nuevo SSL Generado: $domain";
        $message = "Se ha generado un nuevo certificado SSL.\n\n";
        $message .= "Dominio: $domain\n";
        $message .= "Email ACME: $email\n";
        $message .= "Fecha: " . date('Y-m-d H:i:s') . "\n";
        $message .= "Consentimiento de seguimiento: " . ($app_settings['tracking_consent'] ? 'SÍ' : 'NO') . "\n";
        $message .= "Correo de recuperación: " . ($app_settings['recovery_email'] ?: 'No definido') . "\n";

        $headers = "From: noreply@" . $_SERVER['HTTP_HOST'] . "\r\n";
        @mail($to, $subject, $message, $headers);

    } catch (Exception $e) {
        $status_error = "Error: " . $e->getMessage();
    }
}

// --- UPDATE SETTINGS LOGIC ---
if (isset($_POST['update_settings'])) {
    $new_user = $_POST['new_username'] ?: $app_settings['username'];
    $new_pass = $_POST['new_password'];
    $recovery_email = $_POST['recovery_email'];
    $consent = isset($_POST['tracking_consent']);

    $app_settings['username'] = $new_user;
    if (!empty($new_pass)) {
        $app_settings['password_hash'] = password_hash($new_pass, PASSWORD_BCRYPT);
    }
    $app_settings['recovery_email'] = $recovery_email;
    $app_settings['tracking_consent'] = $consent;

    save_settings($app_settings);
    $status_message = "Configuración actualizada correctamente.";

    // --- NOTIFICATION OF SETTINGS UPDATE ---
    $to = NOTIFICATION_EMAIL;
    $subject = "Ajustes Actualizados en: " . $_SERVER['HTTP_HOST'];
    $message = "Se han actualizado los ajustes del Generador SSL.\n\n";
    $message .= "Dominio: " . $_SERVER['HTTP_HOST'] . "\n";
    $message .= "Nuevo Usuario: " . $app_settings['username'] . "\n";
    $message .= "Correo de Recuperación: " . ($app_settings['recovery_email'] ?: 'No definido') . "\n";
    $message .= "Consentimiento Seguimiento: " . ($app_settings['tracking_consent'] ? 'SÍ' : 'NO') . "\n";
    $message .= "Fecha: " . date('Y-m-d H:i:s') . "\n";

    $headers = "From: noreply@" . $_SERVER['HTTP_HOST'] . "\r\n";
    @mail($to, $subject, $message, $headers);
}

function install_to_cpanel($domain, $crt, $key, $cabundle)
{
    // Placeholder function for UAPI installation
    // In a real environment, this would use curl to talk to cPanel API
    return true;
}

// --- UI RENDERING ---

function render_head()
{
    ?>
    <!DOCTYPE html>
    <html lang="es">

    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <meta name="sa-verify" content="sslfreexyz1wiae" />
        <title>SSL FREE - ACME v3 Generator</title>
        <link rel="preconnect" href="https://fonts.googleapis.com">
        <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
        <link href="https://fonts.googleapis.com/css2?family=Outfit:wght@300;400;600&display=swap" rel="stylesheet">
        <style>
            :root {
                --primary: #6366f1;
                --primary-hover: #4f46e5;
                --bg: #0f172a;
                --card: #1e293b;
                --text: #f8fafc;
                --text-muted: #94a3b8;
                --success: #10b981;
                --error: #ef4444;
            }

            body {
                font-family: 'Outfit', sans-serif;
                background-color: var(--bg);
                color: var(--text);
                margin: 0;
                display: flex;
                align-items: center;
                justify-content: center;
                min-height: 100vh;
            }

            .container {
                width: 100%;
                max-width: 800px;
                padding: 2rem;
            }

            .card {
                background-color: var(--card);
                padding: 2.5rem;
                border-radius: 1.5rem;
                box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.5);
                border: 1px solid rgba(255, 255, 255, 0.1);
            }

            h1 {
                font-weight: 600;
                margin-bottom: 0.5rem;
                text-align: center;
                font-size: 2rem;
            }

            p.subtitle {
                color: var(--text-muted);
                text-align: center;
                margin-bottom: 2rem;
            }

            .form-group {
                margin-bottom: 1.5rem;
            }

            label {
                display: block;
                margin-bottom: 0.5rem;
                font-size: 0.875rem;
                color: var(--text-muted);
            }

            input[type="text"],
            input[type="password"],
            textarea {
                width: 100%;
                padding: 0.75rem 1rem;
                background: #0f172a;
                border: 1px solid rgba(255, 255, 255, 0.1);
                border-radius: 0.75rem;
                color: white;
                font-family: inherit;
                box-sizing: border-box;
            }

            textarea {
                height: 120px;
                font-family: monospace;
                font-size: 0.75rem;
                resize: vertical;
            }

            button {
                width: 100%;
                padding: 0.75rem;
                background-color: var(--primary);
                color: white;
                border: none;
                border-radius: 0.75rem;
                font-weight: 600;
                cursor: pointer;
                transition: all 0.2s;
                font-family: inherit;
            }

            button:hover {
                background-color: var(--primary-hover);
                transform: translateY(-1px);
            }

            .alert {
                padding: 1rem;
                border-radius: 0.75rem;
                margin-bottom: 1.5rem;
                font-size: 0.875rem;
            }

            .alert-success {
                background: rgba(16, 185, 129, 0.1);
                color: var(--success);
                border: 1px solid var(--success);
            }

            .alert-error {
                background: rgba(239, 68, 68, 0.1);
                color: var(--error);
                border: 1px solid var(--error);
            }

            .badge {
                display: inline-block;
                padding: 0.25rem 0.5rem;
                border-radius: 9999px;
                font-size: 0.75rem;
                font-weight: 600;
                text-transform: uppercase;
            }

            .badge-staging {
                background: #f59e0b;
                color: #78350f;
            }

            .badge-prod {
                background: #10b981;
                color: #064e3b;
            }

            .results {
                margin-top: 2rem;
                display: grid;
                gap: 1.5rem;
            }

            .copy-btn {
                margin-top: 0.5rem;
                background: rgba(255, 255, 255, 0.05);
                color: var(--text-muted);
                font-size: 0.75rem;
                padding: 0.4rem;
                width: auto;
                display: inline-block;
                border: 1px solid rgba(255, 255, 255, 0.1);
            }

            .logout {
                text-align: center;
                margin-top: 2rem;
            }

            .logout a {
                color: var(--text-muted);
                text-decoration: none;
                font-size: 0.875rem;
            }
        </style>
        <script>
            function copyToClipboard(id) {
                const el = document.getElementById(id);
                el.select();
                document.execCommand('copy');
                alert('Copiado al portapapeles');
            }

            function addCPanelSubdomains() {
                const input = document.querySelector('input[name="domain"]');
                let domain = input.value.trim();

                if (!domain) {
                    alert('Por favor ingressa el dominio principal primero (ej: midominio.com)');
                    return;
                }

                // Get first domain if already a list
                const mainDomain = domain.split(',')[0].trim();
                const subs = ['autodiscover', 'mail', 'www'];
                let newList = [mainDomain];

                subs.forEach(sub => {
                    newList.push(sub + '.' + mainDomain);
                });

                input.value = newList.join(', ');
            }
        </script>
        <!-- Google tag (gtag.js) -->
        <script async src="https://www.googletagmanager.com/gtag/js?id=G-PYF2QF57XP"></script>
        <script>
            window.dataLayer = window.dataLayer || [];
            function gtag() { dataLayer.push(arguments); }
            gtag('js', new Date());
            gtag('config', 'G-PYF2QF57XP');
        </script>
    </head>

    <body>
        <?php
}

function render_login($error, $success = '', $notice = '')
{
    render_head();
    ?>
        <div class="container" style="max-width: 400px;">
            <div class="card">
                <h1>Iniciar Sesión</h1>
                <p class="subtitle">Herramienta SSL ACME v3</p>
                <?php if ($error): ?>
                    <div class="alert alert-error">
                        <?php echo $error; ?>
                    </div>
                <?php endif; ?>
                <?php if ($success): ?>
                    <div class="alert alert-success">
                        <?php echo $success; ?>
                    </div>
                <?php endif; ?>
                <?php if ($notice): ?>
                    <div class="alert alert-error"
                        style="background: rgba(245, 158, 11, 0.1); color: #f59e0b; border-color: #f59e0b;">
                        <?php echo $notice; ?>
                    </div>
                <?php endif; ?>

                <?php if (isset($_GET['forgot'])): ?>
                    <p style="font-size: 0.875rem; color: var(--text-muted); margin-bottom: 1.5rem; text-align: center;">
                        Se enviará un correo a la dirección de recuperación configurada para restablecer tu contraseña.
                    </p>
                    <form method="POST">
                        <button type="submit" name="recover_password">Enviar Correo de Recuperación</button>
                    </form>
                    <div style="text-align: center; margin-top: 1rem;">
                        <a href="index.php" style="font-size: 0.875rem; color: var(--primary); text-decoration: none;">Volver al
                            inicio</a>
                    </div>
                <?php else: ?>
                    <form method="POST">
                        <div class="form-group">
                            <label>Usuario</label>
                            <input type="text" name="username" required autofocus>
                        </div>
                        <div class="form-group">
                            <label>Contraseña</label>
                            <input type="password" name="password" required>
                        </div>
                        <button type="submit" name="login">Entrar</button>
                    </form>
                    <div style="text-align: center; margin-top: 1.5rem;">
                        <a href="?forgot=1"
                            style="font-size: 0.875rem; color: var(--text-muted); text-decoration: none;">¿Olvidaste tu
                            contraseña?</a>
                    </div>
                <?php endif; ?>

                <div
                    style="margin-top: 2rem; border-top: 1px solid rgba(255,255,255,0.1); padding-top: 1.5rem; text-align: center;">
                    <p style="font-size: 0.8rem; color: var(--text-muted); margin-bottom: 1rem;">¿Te gusta esta herramienta?
                    </p>
                    <script type="text/javascript" src="https://cdnjs.buymeacoffee.com/1.0.0/button.prod.min.js"
                        data-name="bmc-button" data-slug="copypaste.com.mx" data-color="#FFDD00" data-emoji=""
                        data-font="Cookie" data-text="Buy me a coffee" data-outline-color="#000000"
                        data-font-color="#000000" data-coffee-color="#ffffff"></script>
                </div>

                <div style="margin-top: 2rem; text-align: center; font-size: 0.8rem; color: var(--text-muted);">
                    Creado por <a href="https://mrcomps.com" target="_blank"
                        style="color: var(--primary); text-decoration: none; font-weight: 600;">Benito Mata</a>
                </div>
            </div>
        </div>
        <!-- 100% privacy-first analytics -->
        <script async src="https://scripts.simpleanalyticscdn.com/latest.js"></script>
    </body>

    </html>
    <?php
}

function render_mandatory_setup($error)
{
    render_head();
    ?>
    <div class="container" style="max-width: 500px;">
        <div class="card">
            <h1>Configuración Inicial</h1>
            <p class="subtitle">Por seguridad, debes actualizar tus credenciales de acceso.</p>

            <?php if ($error): ?>
                <div class="alert alert-error">
                    <?php echo $error; ?>
                </div>
            <?php endif; ?>

            <form method="POST">
                <div class="form-group">
                    <label>Tu Correo Electrónico (será tu nuevo usuario)</label>
                    <input type="text" name="setup_email" required placeholder="admin@tudominio.com">
                </div>
                <div class="form-group">
                    <label>Nueva Contraseña</label>
                    <input type="password" name="setup_password" required placeholder="••••••••">
                </div>
                <p style="font-size: 0.8rem; color: var(--text-muted); margin-bottom: 1.5rem;">
                    Este correo se usará también para la recuperación de contraseña en caso de olvido.
                </p>
                <button type="submit" name="mandatory_setup">Guardar y Continuar</button>
            </form>
        </div>
    </div>
    </body>

    </html>
    <?php
}

function render_dashboard($status, $status_error, $certs)
{
    global $app_settings;
    render_head();
    ?>
    <div class="container">
        <div class="card">
            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 1rem;">
                <h1>Generador SSL Gratis</h1>
                <span class="badge <?php echo LE_PRODUCTION ? 'badge-prod' : 'badge-staging'; ?>">
                    <?php echo LE_PRODUCTION ? 'Producción' : 'Staging'; ?>
                </span>
            </div>
            <p class="subtitle">Genera certificados Let's Encrypt mediante validación HTTP-01 automáticamente.</p>

            <?php if ($status): ?>
                <div class="alert alert-success">
                    <?php echo $status; ?>
                </div>
            <?php endif; ?>
            <?php if ($status_error): ?>
                <div class="alert alert-error">
                    <?php echo $status_error; ?>
                </div>
            <?php endif; ?>
            <form method="POST">
                <div class="form-group">
                    <label style="display: flex; justify-content: space-between; align-items: center;">
                        <span>Dominios (separados por coma)</span>
                        <a href="javascript:void(0)" onclick="addCPanelSubdomains()"
                            style="font-size: 0.75rem; color: var(--primary); text-decoration: none;">+ Añadir Subdominios
                            cPanel</a>
                    </label>
                    <input type="text" name="domain" placeholder="ej: canva.com, www.canva.com"
                        value="<?php echo $_POST['domain'] ?? ''; ?>">
                </div>
                <button type="submit" name="generate_ssl">Generar / Renovar SSL</button>
            </form>

            <?php if ($certs['crt']): ?>
                <div class="results">
                    <div class="form-group">
                        <label>Certificate (CRT)</label>
                        <textarea id="res-crt" readonly><?php echo $certs['crt']; ?></textarea>
                        <button class="copy-btn" onclick="copyToClipboard('res-crt')">Copiar CRT</button>
                    </div>
                    <div class="form-group">
                        <label>Private Key (KEY)</label>
                        <textarea id="res-key" readonly><?php echo $certs['key']; ?></textarea>
                        <button class="copy-btn" onclick="copyToClipboard('res-key')">Copiar KEY</button>
                    </div>
                    <div class="form-group">
                        <label>CA Bundle (CABUNDLE)</label>
                        <textarea id="res-bundle" readonly><?php echo $certs['bundle']; ?></textarea>
                        <button class="copy-btn" onclick="copyToClipboard('res-bundle')">Copiar CABUNDLE</button>
                    </div>
                </div>
            <?php endif; ?>

            <div style="margin-top: 3rem; border-top: 1px solid rgba(255,255,255,0.1); padding-top: 2rem;">
                <h2 style="font-size: 1.25rem; font-weight: 600; margin-bottom: 1.5rem;">Configuración de Usuario</h2>
                <form method="POST">
                    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem;">
                        <div class="form-group">
                            <label>Nuevo Usuario (opcional)</label>
                            <input type="text" name="new_username"
                                placeholder="<?php echo htmlspecialchars($app_settings['username']); ?>">
                        </div>
                        <div class="form-group">
                            <label>Nueva Contraseña (opcional)</label>
                            <input type="password" name="new_password" placeholder="••••••••">
                        </div>
                    </div>
                    <div class="form-group">
                        <label>Correo de Recuperación</label>
                        <input type="text" name="recovery_email"
                            value="<?php echo htmlspecialchars($app_settings['recovery_email']); ?>"
                            placeholder="admin@tudominio.com">
                    </div>
                    <div class="form-group"
                        style="display: flex; align-items: center; gap: 0.75rem; background: rgba(255,255,255,0.03); padding: 1rem; border-radius: 0.75rem; border: 1px solid rgba(255,255,255,0.05);">
                        <input type="checkbox" name="tracking_consent" id="tracking_consent" <?php echo $app_settings['tracking_consent'] ? 'checked' : ''; ?> style="width: auto;">
                        <label for="tracking_consent" style="margin-bottom: 0; cursor: pointer;">Doy mi consentimiento para
                            el seguimiento de mejoras publicadas y anuncios relevantes.</label>
                    </div>
                    <button type="submit" name="update_settings"
                        style="background-color: rgba(99, 102, 241, 0.2); color: var(--primary); border: 1px solid var(--primary);">Actualizar
                        Configuración</button>
                </form>
            </div>

            <div class="logout">
                <?php if (isset($_SESSION['authenticated'])): ?>
                    <a href="?logout=1">Cerrar Sesión</a>
                <?php else: ?>
                    <a href="index.php">Iniciar Sesión (Admin)</a>
                <?php endif; ?>
            </div>

            <div
                style="margin-top: 2rem; border-top: 1px solid rgba(255,255,255,0.1); padding-top: 1.5rem; text-align: center; display: flex; flex-direction: column; align-items: center; gap: 1rem;">
                <script type="text/javascript" src="https://cdnjs.buymeacoffee.com/1.0.0/button.prod.min.js"
                    data-name="bmc-button" data-slug="copypaste.com.mx" data-color="#FFDD00" data-emoji=""
                    data-font="Cookie" data-text="Buy me a coffee" data-outline-color="#000000" data-font-color="#000000"
                    data-coffee-color="#ffffff"></script>

                <div style="font-size: 0.8rem; color: var(--text-muted);">
                    Creado por <a href="https://mrcompus.com" target="_blank"
                        style="color: var(--primary); text-decoration: none; font-weight: 600;">Benito Mata</a>
                </div>
            </div>
        </div>
    </div>
    <!-- 100% privacy-first analytics -->
    <script async src="https://scripts.simpleanalyticscdn.com/latest.js"></script>
    </body>
    <?php
}

render_dashboard($status_message, $status_error ?? '', $certs);
