<?php
/**
 * Consolidated ACMECert Library for ACME v2
 * Based on skoerfgen/ACMECert
 */

namespace skoerfgen\ACMECert;

use Exception;
use stdClass;

class ACME_Exception extends Exception
{
    private $type, $subproblems;
    function __construct($type, $detail, $subproblems = array())
    {
        $this->type = $type;
        $this->subproblems = $subproblems;
        parent::__construct($detail . ' (' . $type . ')');
    }
    function getType()
    {
        return $this->type;
    }
    function getSubproblems()
    {
        return $this->subproblems;
    }
}

class ACMEv2
{
    protected $directories = array(
        'live' => 'https://acme-v02.api.letsencrypt.org/directory',
        'staging' => 'https://acme-staging-v02.api.letsencrypt.org/directory'
    ), $ch = null, $logger = true, $bits, $sha_bits, $directory, $resources, $jwk_header, $kid_header, $account_key, $thumbprint, $nonce = null, $delay_until = null;

    public function __construct($live = true)
    {
        if (is_bool($live)) {
            $this->directory = $this->directories[$live ? 'live' : 'staging'];
        } else {
            $this->directory = $live;
        }
    }

    public function __destruct()
    {
        if (PHP_MAJOR_VERSION < 8 && $this->account_key)
            openssl_pkey_free($this->account_key);
        if ($this->ch)
            curl_close($this->ch);
    }

    public function loadAccountKey($account_key_pem)
    {
        if (PHP_MAJOR_VERSION < 8 && $this->account_key)
            openssl_pkey_free($this->account_key);
        if (false === ($this->account_key = openssl_pkey_get_private($account_key_pem))) {
            throw new Exception('Could not load account key: ' . $this->get_openssl_error());
        }
        if (false === ($details = openssl_pkey_get_details($this->account_key))) {
            throw new Exception('Could not get account key details: ' . $this->get_openssl_error());
        }
        $this->bits = $details['bits'];
        switch ($details['type']) {
            case OPENSSL_KEYTYPE_EC:
                if (version_compare(PHP_VERSION, '7.1.0') < 0)
                    throw new Exception('PHP >= 7.1.0 required for EC keys !');
                $this->sha_bits = ($this->bits == 521 ? 512 : $this->bits);
                $this->jwk_header = array('alg' => 'ES' . $this->sha_bits, 'jwk' => array('crv' => 'P-' . $details['bits'], 'kty' => 'EC', 'x' => $this->base64url(str_pad($details['ec']['x'], ceil($this->bits / 8), "\x00", STR_PAD_LEFT)), 'y' => $this->base64url(str_pad($details['ec']['y'], ceil($this->bits / 8), "\x00", STR_PAD_LEFT))));
                break;
            case OPENSSL_KEYTYPE_RSA:
                $this->sha_bits = 256;
                $this->jwk_header = array('alg' => 'RS256', 'jwk' => array('e' => $this->base64url($details['rsa']['e']), 'kty' => 'RSA', 'n' => $this->base64url($details['rsa']['n'])));
                break;
            default:
                throw new Exception('Unsupported key type! Must be RSA or EC key.');
        }
        $this->kid_header = array('alg' => $this->jwk_header['alg'], 'kid' => null);
        $this->thumbprint = $this->base64url(hash('sha256', json_encode($this->jwk_header['jwk']), true));
    }

    public function getAccountID()
    {
        if (!$this->kid_header['kid'])
            $this->getAccount();
        return $this->kid_header['kid'];
    }

    public function setLogger($value = true)
    {
        $this->logger = $value;
    }
    public function log($txt)
    {
        if ($this->logger === true)
            error_log($txt);
        elseif (is_callable($this->logger))
            ($this->logger)($txt);
    }

    protected function create_ACME_Exception($type, $detail, $subproblems = array())
    {
        $this->log('ACME_Exception: ' . $detail . ' (' . $type . ')');
        return new ACME_Exception($type, $detail, $subproblems);
    }

    protected function get_openssl_error()
    {
        $out = array();
        $arr = error_get_last();
        if (is_array($arr))
            $out[] = $arr['message'];
        $out[] = openssl_error_string();
        return implode(' | ', $out);
    }

    protected function getAccount()
    {
        return $this->request('newAccount', array('onlyReturnExisting' => true));
    }

    protected function keyAuthorization($token)
    {
        return $token . '.' . $this->thumbprint;
    }

    protected function readDirectory()
    {
        $ret = $this->http_request($this->directory);
        if (!is_array($ret['body']) || !empty(array_diff_key(array_flip(array('newNonce', 'newAccount', 'newOrder')), $ret['body']))) {
            throw new Exception('Failed to read directory: ' . $this->directory);
        }
        $this->resources = $ret['body'];
    }

    protected function request($type, $payload = '', $retry = false)
    {
        if (!$this->jwk_header)
            throw new Exception('use loadAccountKey to load an account key');
        if (!$this->resources)
            $this->readDirectory();
        if (0 === stripos($type, 'http')) {
            $this->resources['_tmp'] = $type;
            $type = '_tmp';
        }
        try {
            $ret = $this->http_request($this->resources[$type], json_encode($this->jws_encapsulate($type, $payload)));
        } catch (ACME_Exception $e) {
            if (!$retry && $e->getType() === 'urn:ietf:params:acme:error:badNonce')
                return $this->request($type, $payload, true);
            throw $e;
        }
        if (!$this->kid_header['kid'] && $type === 'newAccount')
            $this->kid_header['kid'] = $ret['headers']['location'];
        return $ret;
    }

    protected function jws_encapsulate($type, $payload, $is_inner_jws = false)
    {
        if ($type === 'newAccount' || $is_inner_jws)
            $protected = $this->jwk_header;
        else {
            $this->getAccountID();
            $protected = $this->kid_header;
        }
        if (!$is_inner_jws) {
            if (!$this->nonce)
                $this->http_request($this->resources['newNonce'], false);
            $protected['nonce'] = $this->nonce;
            $this->nonce = null;
        }
        $protected['url'] = $this->resources[$type];
        $protected64 = $this->base64url(json_encode($protected, JSON_UNESCAPED_SLASHES));
        $payload64 = $this->base64url(is_string($payload) ? $payload : json_encode($payload, JSON_UNESCAPED_SLASHES));
        if (false === openssl_sign($protected64 . '.' . $payload64, $signature, $this->account_key, 'SHA' . $this->sha_bits))
            throw new Exception('Failed to sign payload !');
        return array('protected' => $protected64, 'payload' => $payload64, 'signature' => $this->base64url($this->jwk_header['alg'][0] == 'R' ? $signature : $this->asn2signature($signature, ceil($this->bits / 8))));
    }

    private function asn2signature($asn, $pad_len)
    {
        $asn = substr($asn, $asn[1] === "\x81" ? 3 : 2);
        $R = ltrim(substr($asn, 2, ord($asn[1])), "\x00");
        $asn = substr($asn, ord($asn[1]) + 2);
        $S = ltrim(substr($asn, 2, ord($asn[1])), "\x00");
        return str_pad($R, $pad_len, "\x00", STR_PAD_LEFT) . str_pad($S, $pad_len, "\x00", STR_PAD_LEFT);
    }

    protected function base64url($data)
    {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }
    protected function base64url_decode($data)
    {
        return base64_decode(strtr($data, '-_', '+/'));
    }
    private function json_decode($str)
    {
        $ret = json_decode($str, true);
        if ($ret === null)
            throw new Exception('Could not parse JSON: ' . $str);
        return $ret;
    }

    protected function http_request($url, $data = null)
    {
        if ($this->ch === null) {
            if (extension_loaded('curl') && $this->ch = curl_init())
                ;
            elseif (ini_get('allow_url_fopen'))
                $this->ch = false;
            else
                throw new Exception('Can not connect, no cURL or fopen wrappers enabled !');
        }
        $method = $data === false ? 'HEAD' : ($data === null ? 'GET' : 'POST');
        $header = ($data === null || $data === false) ? array() : array('Content-Type: application/jose+json');
        if ($this->ch) {
            $headers = array();
            curl_setopt_array($this->ch, array(
                CURLOPT_URL => $url,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_TCP_NODELAY => true,
                CURLOPT_NOBODY => $data === false,
                CURLOPT_CUSTOMREQUEST => $method,
                CURLOPT_HTTPHEADER => $header,
                CURLOPT_POSTFIELDS => $data,
                CURLOPT_HEADERFUNCTION => function ($ch, $header) use (&$headers) {
                    $headers[] = $header;
                    return strlen($header);
                }
            ));
            $body = curl_exec($this->ch);
            if ($body === false)
                throw new Exception('HTTP Request Error: ' . curl_error($this->ch));
        } else {
            $body = file_get_contents($url, false, stream_context_create(array('http' => array('header' => $header, 'method' => $method, 'ignore_errors' => true, 'timeout' => 60, 'content' => $data))));
            if ($body === false)
                throw new Exception('HTTP Request Error');
            $headers = $http_response_header;
        }
        $headers = array_reduce(array_filter($headers, function ($item) {
            return trim($item) != '';
        }), function ($carry, $item) use (&$code) {
            $parts = explode(':', $item, 2);
            if (count($parts) === 1) {
                list(, $code) = explode(' ', trim($item), 3);
                $carry = array();
            } else {
                list($k, $v) = $parts;
                $k = strtolower(trim($k));
                $carry[$k] = trim($v);
            }
            return $carry;
        }, array());
        if (!empty($headers['replay-nonce']))
            $this->nonce = $headers['replay-nonce'];
        if (isset($headers['content-type'])) {
            if ($headers['content-type'] === 'application/problem+json' || ($headers['content-type'] === 'application/json' && $code[0] != '2')) {
                $this->handleError($this->json_decode($body));
            } elseif ($headers['content-type'] === 'application/json')
                $body = $this->json_decode($body);
        }
        if ($code[0] != '2')
            throw new Exception('Invalid HTTP status: ' . $code);
        return array('code' => $code, 'headers' => $headers, 'body' => $body);
    }

    private function handleError($error)
    {
        throw $this->create_ACME_Exception($error['type'], $error['detail']);
    }
}

class ACMECert extends ACMEv2
{
    public function register($termsOfServiceAgreed = false, $contacts = array())
    {
        $ret = $this->request('newAccount', array(
            'termsOfServiceAgreed' => (bool) $termsOfServiceAgreed,
            'contact' => array_map(function ($c) {
                return 'mailto:' . $c;
            }, (array) $contacts)
        ));
        return $ret['body'];
    }

    public function getCertificateChain($pem, $domains, $callback)
    {
        $ret = $this->request('newOrder', array(
            'identifiers' => array_map(function ($d) {
                return array('type' => 'dns', 'value' => $d);
            }, $domains)
        ));
        $order = $ret['body'];
        $order_location = $ret['headers']['location'];

        foreach ($order['authorizations'] as $auth_url) {
            $ret = $this->request($auth_url, '');
            $auth = $ret['body'];
            if ($auth['status'] === 'valid')
                continue;
            foreach ($auth['challenges'] as $challenge) {
                if ($challenge['type'] != 'http-01')
                    continue;
                $callback(array('domain' => $auth['identifier']['value'], 'key' => '/.well-known/acme-challenge/' . $challenge['token'], 'value' => $challenge['token'] . '.' . $this->thumbprint));
                $this->request($challenge['url'], new stdClass);
                if (!$this->poll('pending', $auth_url, $ret))
                    throw new Exception('Validation failed for domain: ' . $auth['identifier']['value']);
            }
        }

        $private_key = openssl_pkey_get_private($pem);
        $dn = array('commonName' => $domains[0]);

        $configargs = array(
            'digest_alg' => 'sha256',
            'config' => null, // Use system default config
        );

        if (count($domains) > 1) {
            $san = "DNS:" . implode(",DNS:", $domains);
            // We need a temporary config file for SAN in openssl_csr_new if not using a custom config string
            // However, we can also pass extensions via configargs if we provide a custom config content
            $tmp_config = tmpfile();
            $tmp_config_path = stream_get_meta_data($tmp_config)['uri'];
            $config_content = "[req]\ndistinguished_name=req_distinguished_name\nreq_extensions=v3_req\n[req_distinguished_name]\n[v3_req]\nsubjectAltName=" . $san;
            file_put_contents($tmp_config_path, $config_content);
            $configargs['config'] = $tmp_config_path;
        }

        $csr_res = openssl_csr_new($dn, $private_key, $configargs);
        openssl_csr_export($csr_res, $csr);

        $this->request($order['finalize'], array('csr' => $this->base64url($this->pem2der($csr))));
        $this->poll('processing', $order_location, $ret);

        $ret = $this->request($ret['certificate'], '');
        return $ret['body'];
    }

    private function poll($initial, $url, &$ret)
    {
        for ($i = 0; $i < 10; $i++) {
            $ret = $this->request($url);
            $ret = $ret['body'];
            if ($ret['status'] !== $initial)
                return $ret['status'] === 'valid' || $ret['status'] === 'ready';
            sleep(pow(2, min($i, 5)));
        }
        return false;
    }

    private function pem2der($pem)
    {
        return base64_decode(implode('', array_slice(array_map('trim', explode("\n", trim($pem))), 1, -1)));
    }

    public function generateRSAKey($bits = 2048)
    {
        $key = openssl_pkey_new(array('private_key_bits' => (int) $bits, 'private_key_type' => OPENSSL_KEYTYPE_RSA));
        openssl_pkey_export($key, $pem);
        return $pem;
    }
}
